<?php
/*
 * Xibo - Digital Signage - http://www.xibo.org.uk
 * Copyright (C) 2011-13 Daniel Garner
 *
 * This file is part of Xibo.
 *
 * Xibo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * Xibo is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Xibo.  If not, see <http://www.gnu.org/licenses/>.
 */
defined('XIBO') or die("Sorry, you are not allowed to directly access this page.<br /> Please press the back button in your browser.");

class DataSet extends Data
{
    /**
     * Add a data set
     * @param <type> $dataSet
     * @param <type> $description
     * @param <type> $userId
     * @return <type>
     */
    public function Add($dataSet, $description, $userId)
    {
        try {
            $dbh = PDOConnect::init();

            // Validation
            if (strlen($dataSet) > 50 || strlen($dataSet) < 1)
                return $this->SetError(25001, __("Name must be between 1 and 50 characters"));

            if (strlen($description) > 254)
                return $this->SetError(25002, __("Description can not be longer than 254 characters"));


            // Ensure there are no layouts with the same name
            $sth = $dbh->prepare('SELECT DataSet FROM dataset WHERE DataSet = :dataset');
            $sth->execute(array(
                    'dataset' => $dataSet
                ));

            if ($row = $sth->fetch())
                return $this->SetError(25004, sprintf(__("There is already dataset called '%s'. Please choose another name."), $dataSet));

            // End Validation

            $SQL = "INSERT INTO dataset (DataSet, Description, UserID) ";
            $SQL .= " VALUES (:dataset, :description, :userid) ";

            // Insert the data set
            $sth = $dbh->prepare($SQL);
            $sth->execute(array(
                    'dataset' => $dataSet,
                    'description' => $description,
                    'userid' => $userId
                ));

            $id = $dbh->lastInsertId();

            Debug::LogEntry('audit', 'Complete', 'DataSet', 'Add');

            return $id;
        }
        catch (Exception $e) {
            Debug::LogEntry('error', $e->getMessage());
            return $this->SetError(25005, __('Could not add DataSet'));
        }
    }

    /**
     * Edit a DataSet
     * @param <type> $dataSetId
     * @param <type> $dataSet
     * @param <type> $description
     */
    public function Edit($dataSetId, $dataSet, $description)
    {
        try {
            $dbh = PDOConnect::init();

            // Validation
            if (strlen($dataSet) > 50 || strlen($dataSet) < 1)
            {
                $this->SetError(25001, __("Name must be between 1 and 50 characters"));
                return false;
            }

            if (strlen($description) > 254)
            {
                $this->SetError(25002, __("Description can not be longer than 254 characters"));
                return false;
            }

            // Ensure there are no layouts with the same name
            $sth = $dbh->prepare('SELECT DataSet FROM dataset WHERE DataSet = :dataset AND DataSetID <> :datasetid');
            $sth->execute(array(
                    'dataset' => $dataSet,
                    'datasetid' => $dataSetId
                ));

            if ($row = $sth->fetch())
                return $this->SetError(25004, sprintf(__("There is already dataset called '%s'. Please choose another name."), $dataSet));

            // End Validation
             
            // Update the data set
            $sth = $dbh->prepare('UPDATE dataset SET DataSet = :dataset, Description = :description WHERE DataSetID = :datasetid');
            $sth->execute(array(
                    'dataset' => $dataSet,
                    'description' => $description,
                    'datasetid' => $dataSetId
                ));

            return true;
        }
        catch (Exception $e) {
            Debug::LogEntry('error', $e->getMessage());
            return $this->SetError(25005, sprintf(__('Cannot edit dataset %s'), $dataSet));
        }
    }

    /**
     * Delete DataSet
     * @param <type> $dataSetId
     */
    public function Delete($dataSetId)
    {
        try {
            $dbh = PDOConnect::init();

            // First check to see if we have any data
            $sth = $dbh->prepare('SELECT * FROM `datasetdata` INNER JOIN `datasetcolumn` ON datasetcolumn.DataSetColumnID = datasetdata.DataSetColumnID WHERE datasetcolumn.DataSetID = :datasetid');
            $sth->execute(array(
                    'datasetid' => $dataSetId
                ));

            if ($row = $sth->fetch())
                return $this->SetError(25005, __('There is data assigned to this data set, cannot delete.'));
            
            // Delete security
            Kit::ClassLoader('datasetgroupsecurity');
            $security = new DataSetGroupSecurity($this->db);
            $security->UnlinkAll($dataSetId);

            // Delete columns
            $dataSetObject = new DataSetColumn($this->db);
            if (!$dataSetObject->DeleteAll($dataSetId))
                return $this->SetError(25005, __('Cannot delete dataset, columns could not be deleted.'));

            // Delete data set
            $sth = $dbh->prepare('DELETE FROM dataset WHERE DataSetID = :datasetid');
            $sth->execute(array(
                    'datasetid' => $dataSetId
                ));

            return true;
        }
        catch (Exception $e) {

            Debug::LogEntry('error', $e->getMessage());

            if (!$this->IsError())
                $this->SetError(25005, sprintf(__('Cannot edit dataset %s'), $dataSet));

            return false;
        }
    }

    /**
     * Data Set Results
     * @param <type> $dataSetId
     * @param <type> $columnIds
     * @param <type> $filter
     * @param <type> $ordering
     * @param <type> $lowerLimit
     * @param <type> $upperLimit
     * @return <type>
     */
    public function DataSetResults($dataSetId, $columnIds, $filter = '', $ordering = '', $lowerLimit = 0, $upperLimit = 0, $displayId = 0, $associative = false)
    {
        $db =& $this->db;

        $selectSQL = '';
        $outserSelect = '';
        $finalSelect = '';
        $results = array();
        $headings = array();
        
        $columns = explode(',', $columnIds);

        // Get the Latitude and Longitude ( might be used in a formula )
        if ($displayId == 0) {
            $defaultLat = Config::GetSetting('DEFAULT_LAT');
            $defaultLong = Config::GetSetting('DEFAULT_LONG');
            $displayGeoLocation = "GEOMFROMTEXT('POINT(" . $defaultLat . " " . $defaultLong . ")')";
        }
        else
            $displayGeoLocation = sprintf("(SELECT GeoLocation FROM `display` WHERE DisplayID = %d)", $displayId);

        // Get all columns for the cross tab
        $allColumns = $db->GetArray(sprintf('SELECT DataSetColumnID, Heading, DataSetColumnTypeID, Formula FROM datasetcolumn WHERE DataSetID = %d' , $dataSetId));

        foreach($allColumns as $col)
        {
            $heading = $col;
            $heading['Text'] = $heading['Heading'];
            
            // Is this column a formula column or a value column?
            if ($col['DataSetColumnTypeID'] == 2) {
                // Formula
                $heading['Heading'] = str_replace('[DisplayGeoLocation]', $displayGeoLocation, $col['Formula']) . ' AS ' . $heading['Heading'];
            }
            else {
                // Value
                $selectSQL .= sprintf("MAX(CASE WHEN DataSetColumnID = %d THEN `Value` ELSE null END) AS '%s', ", $col['DataSetColumnID'], $heading['Heading']);
            }

            $headings[] = $heading;
        }

        // Build our select statement including formulas
        foreach($headings as $heading)
        {
            if ($heading['DataSetColumnTypeID'] == 2)
                // This is a formula, so the heading has been morphed into some SQL to run
                $outserSelect .= sprintf(' %s,', $heading['Heading']);
            else
                $outserSelect .= sprintf(' `%s`,', $heading['Heading']);
        }
        $outserSelect = rtrim($outserSelect, ',');

        // For each heading, put it in the correct order (according to $columns)
        foreach($columns as $visibleColumn)
        {
            foreach($headings as $heading)
            {
                if ($heading['DataSetColumnID'] == $visibleColumn)
                {
                    $finalSelect .= sprintf(' `%s`,', $heading['Text']);
                    
                    $results['Columns'][] = $heading['Text'];
                }
            }
        }
        $finalSelect = rtrim($finalSelect, ',');

        // We are ready to build the select and from part of the SQL
        $SQL  = "SELECT $finalSelect ";
        $SQL .= "  FROM ( ";
        $SQL .= "   SELECT $outserSelect ,";
        $SQL .= "           RowNumber ";
        $SQL .= "     FROM ( ";
        $SQL .= "      SELECT $selectSQL ";
        $SQL .= "          RowNumber ";
        $SQL .= "        FROM (";
        $SQL .= "          SELECT datasetcolumn.DataSetColumnID, datasetdata.RowNumber, datasetdata.`Value` ";
        $SQL .= "            FROM datasetdata ";
        $SQL .= "              INNER JOIN datasetcolumn ";
        $SQL .= "              ON datasetcolumn.DataSetColumnID = datasetdata.DataSetColumnID ";
        $SQL .= sprintf("       WHERE datasetcolumn.DataSetID = %d ", $dataSetId);
        $SQL .= "          ) datasetdatainner ";
        $SQL .= "      GROUP BY RowNumber ";
        $SQL .= "    ) datasetdata ";
        if ($filter != '')
        {
            $SQL .= ' WHERE ' . $filter;
        }
        $SQL .= ' ) finalselect ';

        if ($ordering != '')
        {
            $order = ' ORDER BY ';

            $ordering = explode(',', $ordering);

            foreach ($ordering as $orderPair)
            {
                if (strripos($orderPair, ' DESC'))
                {
                    $orderPair = str_replace(' DESC', '', $orderPair);
                    $order .= sprintf(" `%s` DESC,", $db->escape_string($orderPair));
                }
                else
                {
                    $order .= sprintf(" `%s`,", $db->escape_string($orderPair));
                }
            }

            $SQL .= trim($order, ',');
        }
        else
        {
            $SQL .= " ORDER BY RowNumber ";
        }

        if ($lowerLimit != 0 || $upperLimit != 0)
        {
            // Lower limit should be 0 based
            if ($lowerLimit != 0)
                $lowerLimit = $lowerLimit - 1;

            // Upper limit should be the distance between upper and lower
            $upperLimit = $upperLimit - $lowerLimit;

            // Substitute in
            $SQL .= sprintf(' LIMIT %d, %d ', $lowerLimit, $upperLimit);
        }

        Debug::LogEntry('audit', $SQL);

        if (!$rows = $db->GetArray($SQL, $associative))
            trigger_error($db->error());

        if (!is_array($rows))
            $rows = array();
            
        $results['Rows'] = $rows;

        return $results;
    }
}
?>